/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Collections.Specialized;
using System.Collections;
using System.Runtime.Serialization;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Interfaces;

namespace Borland.Eco.Persistence
{
	/// <summary>
	/// This class implements a key mapper for autoincremental fields. The actual id values are allocated by the database
	/// </summary>
	public class DefaultEcoIdMapper: GenericKeyMapper, IKeyMapper
	{
		public DefaultEcoIdMapper(): base()
		{
			m_LastReservedId = -2;
			m_NextId = -1;
		}

		public static string DefaultKeyMapperName = "DefaultEcoIdMapper";
		
		private int m_ReservedCount;
		private int m_LastReservedId;
		private int m_NextId;
		private ISingleColumnAttributemapping m_Int32Mapper;
		// not used private bool m_UseInternalNullKey;
		public bool UseInternalNullKey { get { return false; } }
		private static string IDTABLE_NAME = "<Prefix>_ID"; // do not localize
		private static string IDCOLUMN_NAME = "BOLD_ID";  // do not localize
															// this is the column name used in the generatortable.
															// It could change to ECO_ID, but that would break backwards compatibility
		private static int INTERNALNULLKEY = -1;       // used for 'databases' without NULL such as paradox  and DBase

		private string EffectiveIdTableName()
		{
			string res = IDTABLE_NAME.Replace("<Prefix>", SqlDatabaseConfig.SystemTablePrefix); // do not localize
			return res;
		}

		private string InternalColumnType(int columnNum)
		{
			return m_Int32Mapper.ColumnType(columnNum);
		}

		public void IdToParameters(ObjectId id, IDataParameter[] parameters)
		{
			System.Object value;
			if (id != null) 
				value = id.Key;
			else if (UseInternalNullKey)
				value = INTERNALNULLKEY;
			else
				value = null;

			GetKeyMapper(0).ValueToParameter(value, parameters[0]);
		}

		public ObjectId IdFromFields(IField[] keyFields, ArrayList keyMappers, int classId, int timeStamp)
		{
			IField f = keyFields[0];
			if (f.IsNull)
				return null;
			else
			{
				System.Object value = GetKeyMapper(0).ColumnToValue(f.Value);
				int intValue = (System.Int32)value;
				if (intValue == INTERNALNULLKEY)
					return null;
				else if (timeStamp != VersionConst.CurrentVersion)
					return new TimestampedDefaultId(intValue, classId, timeStamp);
				else
					return new DefaultId(intValue, classId);
			}
		}
		public int ColumnCount()
		{
			return 1;
		}

		public void ReserveId()
		{
			m_ReservedCount++;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="db"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="oldId"/> is null.</exception>
		///<exception cref="EcoCantGetNewObjectId">Thrown if there was a problem reserving IDs. The original exception is passed as InnerException</exception>
		public ObjectId NewIdPreUpdate(IDatabase db, ObjectId oldId)
		{
			if (db == null) throw new ArgumentNullException("db"); // Do not localize
			if (oldId == null) throw new ArgumentNullException("oldId"); // Do not localize
			if (m_NextId > m_LastReservedId)
			{
				db.StartTransaction();
				IQuery aQuery = db.GetQuery();
				IExecQuery aExecQuery = db.GetExecQuery();
				try
				{
					try
					{
						aExecQuery.AssignSqlText(String.Format(CultureInfo.InvariantCulture, "UPDATE {0} SET {1} = {2} + {3}",  // do not localize
												  EffectiveIdTableName(),
												  IDCOLUMN_NAME,
												  IDCOLUMN_NAME,
												  m_ReservedCount));
						aExecQuery.ExecSql();
						aQuery.AssignSqlText(String.Format(CultureInfo.InvariantCulture, "SELECT {0} FROM {1}", // do not localize
											   IDCOLUMN_NAME,
											   EffectiveIdTableName()));
						bool idFound = false;
						int newId = -1;
						aQuery.Open();
						try
						{
							if (!aQuery.Eof)
							{
								newId = aQuery[0].AsInt32;
								idFound = true;
							}
						}
						finally
						{
							aQuery.Close();
						}
						
						if (!idFound)
						{
							// the first row in the db was not there, lets add it.
							newId = 1 + m_ReservedCount;

							aExecQuery.AssignSqlText(String.Format(CultureInfo.InvariantCulture, "INSERT INTO {0} ({1}) VALUES ({2})", // do not localize
								EffectiveIdTableName(),
								IDCOLUMN_NAME,
								newId));
							aExecQuery.ExecSql();
						}
						db.Commit();
						m_NextId = newId - m_ReservedCount;
						m_LastReservedId = m_NextId + m_ReservedCount - 1;
					}
					catch (Exception e)
					{
						db.RollBack();
						throw new EcoCantGetNewObjectId(String.Format(CultureInfo.InvariantCulture, 
							InterfacesStringRes.sCantGetID(e.GetType().ToString(), e.ToString()), e));
					}
				}
				finally
				{
					db.ReleaseQuery(aQuery);
					db.ReleaseExecQuery(aExecQuery);
				}
			}
			m_ReservedCount = 0;
			ObjectId res = new DefaultId(m_NextId, oldId.ClassId);

			m_NextId = m_NextId + 1;
			return res;
		}

		public ObjectId NewIdFromAttributes(ObjectContents objContents, ArrayList memberIndexes, IdTranslationList translationList)
		{
			return null;
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="db"/> is null</exception>
		public void InitializePSDescriptions(DatabaseDefinition db)
		{
			if (db == null) throw new ArgumentNullException("db"); // Do not localize
			TableDefinition idTable = db.EnsureTable(EffectiveIdTableName());
			ColumnDefinition column = idTable.EnsureColumn(IDCOLUMN_NAME);
			column.SqlType = InternalColumnType(0);
			column.AllowNull = false;
			column.DefaultValue = "";
		}

		public void InitializeDatabase(IDatabase db)
		{
			// intentionally left blank
		}
		public void InitializeDatabaseScript(StringCollection script)
		{
			// intentionally left blank
		}

		public IdAllocationMode GetIdAllocationMode()
		{
			return IdAllocationMode.PreUpdate;
		}
		public string ColumnType(ArrayList KeyMappers, int columnNum)
		{
			return InternalColumnType(columnNum);
		}

		public void Initialize(SqlDatabaseConfig config, ArrayList keyMappers)
		{
			InternalInitialize(config, keyMappers);
			m_Int32Mapper = MapperDictionary.FindSingleColumnMappingByName(typeof(System.Int32).ToString());
		}

		public string IdListToWhereFragment(ObjectIdCollection idList, IParameterized parameterized, StringCollection idColumns)
		{
			return InternalIdListToWhereFragment(idList, parameterized, idColumns, m_Int32Mapper);
		}
		public string IdToWhereFragment(ObjectId id, IParameterized parameterized, StringCollection idColumns)
		{
			ObjectIdCollection list = new ObjectIdCollection();
			list.Add(id);
			return IdListToWhereFragment(list, parameterized, idColumns);
		}
	}
}